<?php
/**
 * Cron Job: چک کردن مصرف eSIM و ارسال اعلان
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

header('Content-Type: application/json; charset=utf-8');

// Security key
define('CRON_SECRET_KEY', 'esim_cron_2024_secret');

// چک امنیت
if (php_sapi_name() !== 'cli') {
    $providedKey = $_GET['key'] ?? '';
    if ($providedKey !== CRON_SECRET_KEY) {
        http_response_code(403);
        die(json_encode(['error' => 'Access denied']));
    }
}

// لود فایل‌ها
try {
    require_once __DIR__ . '/config/config.php';
} catch (Throwable $e) {
    die(json_encode(['error' => 'Config error: ' . $e->getMessage()]));
}

try {
    require_once __DIR__ . '/includes/functions.php';
} catch (Throwable $e) {
    die(json_encode(['error' => 'Functions error: ' . $e->getMessage()]));
}

try {
    require_once __DIR__ . '/api/esim-api.php';
} catch (Throwable $e) {
    die(json_encode(['error' => 'eSIM API error: ' . $e->getMessage()]));
}

try {
    require_once __DIR__ . '/api/sms-api.php';
} catch (Throwable $e) {
    die(json_encode(['error' => 'SMS API error: ' . $e->getMessage()]));
}

// تنظیمات
$LOW_USAGE_THRESHOLD = 10;
$SITE_NAME = 'اولeSIM';
$results = [];
$errors = [];

try {
    // اتصال دیتابیس
    $conn = getDBConnection();
    
    // چک فیلد usage_alert_sent
    $checkColumn = $conn->query("SHOW COLUMNS FROM orders LIKE 'usage_alert_sent'");
    $hasAlertColumn = ($checkColumn && $checkColumn->num_rows > 0);
    
    // کوئری - بدون u.name چون ممکنه وجود نداشته باشه
    if ($hasAlertColumn) {
        $query = "SELECT o.*, u.mobile, u.email 
                  FROM orders o 
                  LEFT JOIN users u ON o.user_id = u.id 
                  WHERE o.status = 'paid' 
                  AND o.iccid IS NOT NULL 
                  AND o.iccid != ''
                  AND (o.usage_alert_sent IS NULL OR o.usage_alert_sent = 0)
                  ORDER BY o.created_at DESC
                  LIMIT 50";
    } else {
        $query = "SELECT o.*, u.mobile, u.email 
                  FROM orders o 
                  LEFT JOIN users u ON o.user_id = u.id 
                  WHERE o.status = 'paid' 
                  AND o.iccid IS NOT NULL 
                  AND o.iccid != ''
                  ORDER BY o.created_at DESC
                  LIMIT 50";
    }
    
    $result = $conn->query($query);
    if (!$result) {
        throw new Exception('Query error: ' . $conn->error);
    }
    
    $esims = [];
    while ($row = $result->fetch_assoc()) {
        $esims[] = $row;
    }
    $conn->close();
    
    if (empty($esims)) {
        echo json_encode([
            'success' => true,
            'message' => 'No eSIMs to check',
            'checked' => 0,
            'alerts_sent' => 0,
            'has_alert_column' => $hasAlertColumn
        ]);
        exit;
    }
    
    $api = new eSIMAPI();
    $sms = new SmsAPI();
    $alertsSent = 0;
    $checked = 0;
    
    foreach ($esims as $esim) {
        $checked++;
        $esimResult = [
            'order_id' => $esim['id'],
            'mobile' => $esim['mobile'],
            'status' => 'skipped'
        ];
        
        try {
            // گرفتن اطلاعات از API
            $apiResponse = $api->queryProfile($esim['iccid']);
            
            if (!$apiResponse || !isset($apiResponse['obj']['esimList'][0])) {
                $esimResult['status'] = 'api_no_data';
                $results[] = $esimResult;
                continue;
            }
            
            $esimData = $apiResponse['obj']['esimList'][0];
            $totalVolume = $esimData['totalVolume'] ?? 0;
            $usedVolume = $esimData['orderUsage'] ?? 0;
            $esimStatus = $esimData['esimStatus'] ?? '';
            
            $esimResult['esim_status'] = $esimStatus;
            $esimResult['total_volume'] = $totalVolume;
            $esimResult['used_volume'] = $usedVolume;
            
            // فقط eSIM های فعال
            if (!in_array($esimStatus, ['ACTIVE', 'IN_USE', 'ACTIVATED'])) {
                $esimResult['status'] = 'not_active';
                $results[] = $esimResult;
                continue;
            }
            
            if ($totalVolume <= 0) {
                $esimResult['status'] = 'zero_volume';
                $results[] = $esimResult;
                continue;
            }
            
            // محاسبه درصد
            $remainingPercent = round((($totalVolume - $usedVolume) / $totalVolume) * 100, 1);
            $esimResult['remaining_percent'] = $remainingPercent;
            
            // اگر نیاز به اعلان بود
            if ($remainingPercent <= $LOW_USAGE_THRESHOLD) {
                $userMobile = $esim['mobile'];
                $message = "سلام از {$SITE_NAME}\nاز بسته eSIM شما فقط {$remainingPercent}% باقی مانده.\nبرای افزایش حجم به پنل کاربری مراجعه کنید.";
                
                $smsResult = $sms->sendSMS($userMobile, $message);
                
                if ($smsResult['success']) {
                    $esimResult['status'] = 'alert_sent';
                    
                    // علامت‌گذاری
                    if ($hasAlertColumn) {
                        $conn2 = getDBConnection();
                        $stmt = $conn2->prepare("UPDATE orders SET usage_alert_sent = 1, usage_alert_time = NOW() WHERE id = ?");
                        $stmt->bind_param("i", $esim['id']);
                        $stmt->execute();
                        $stmt->close();
                        $conn2->close();
                    }
                    
                    $alertsSent++;
                } else {
                    $esimResult['status'] = 'sms_failed';
                    $esimResult['sms_error'] = $smsResult['message'] ?? 'unknown';
                }
            } else {
                $esimResult['status'] = 'ok';
            }
            
        } catch (Throwable $e) {
            $esimResult['status'] = 'error';
            $esimResult['error'] = $e->getMessage();
            $errors[] = $e->getMessage();
        }
        
        $results[] = $esimResult;
        
        // تاخیر
        usleep(300000);
    }
    
    echo json_encode([
        'success' => true,
        'checked' => $checked,
        'alerts_sent' => $alertsSent,
        'has_alert_column' => $hasAlertColumn,
        'threshold' => $LOW_USAGE_THRESHOLD,
        'results' => $results,
        'errors' => $errors,
        'timestamp' => date('Y-m-d H:i:s')
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    
} catch (Throwable $e) {
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine()
    ]);
}
