/**
 * Token-based Authentication Helper for WebView/Mobile Apps
 * این اسکریپت به طور خودکار token را به تمام لینک‌ها و فرم‌ها اضافه می‌کند
 */

(function() {
    'use strict';
    
    // Get token from localStorage
    function getAuthToken() {
        return localStorage.getItem('auth_token');
    }
    
    // Add token to URL
    function addTokenToUrl(url) {
        const token = getAuthToken();
        if (!token) return url;
        
        try {
            const urlObj = new URL(url, window.location.origin);
            urlObj.searchParams.set('token', token);
            return urlObj.toString();
        } catch (e) {
            // If URL parsing fails, use simple concatenation
            const separator = url.includes('?') ? '&' : '?';
            return url + separator + 'token=' + token;
        }
    }
    
    // Add token to all internal links
    function addTokenToLinks() {
        const token = getAuthToken();
        if (!token) return;
        
        const links = document.querySelectorAll('a[href]');
        links.forEach(function(link) {
            const href = link.getAttribute('href');
            
            // Skip external links, anchors, and javascript links
            if (!href || href.startsWith('#') || href.startsWith('javascript:') || 
                href.startsWith('http://') || href.startsWith('https://')) {
                return;
            }
            
            // Add token to internal links
            link.href = addTokenToUrl(href);
        });
    }
    
    // Add token to all forms
    function addTokenToForms() {
        const token = getAuthToken();
        if (!token) return;
        
        const forms = document.querySelectorAll('form');
        forms.forEach(function(form) {
            // Skip if form already has token input
            if (form.querySelector('input[name="token"]')) {
                return;
            }
            
            // Skip forms that go to external URLs
            const action = form.getAttribute('action');
            if (action && (action.startsWith('http://') || action.startsWith('https://'))) {
                return;
            }
            
            // Add hidden token input
            const tokenInput = document.createElement('input');
            tokenInput.type = 'hidden';
            tokenInput.name = 'token';
            tokenInput.value = token;
            form.appendChild(tokenInput);
        });
    }
    
    // Add token to fetch requests
    const originalFetch = window.fetch;
    window.fetch = function() {
        const token = getAuthToken();
        
        if (token && arguments.length > 0) {
            // Add Authorization header if not already present
            if (arguments.length === 1) {
                arguments[1] = {
                    headers: {
                        'Authorization': 'Bearer ' + token
                    }
                };
            } else if (arguments[1] && !arguments[1].headers) {
                arguments[1].headers = {
                    'Authorization': 'Bearer ' + token
                };
            } else if (arguments[1] && arguments[1].headers && !arguments[1].headers.Authorization) {
                arguments[1].headers.Authorization = 'Bearer ' + token;
            }
        }
        
        return originalFetch.apply(this, arguments);
    };
    
    // Initialize on page load
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', function() {
            addTokenToLinks();
            addTokenToForms();
        });
    } else {
        addTokenToLinks();
        addTokenToForms();
    }
    
    // Re-add token when new content is added (for AJAX)
    const observer = new MutationObserver(function() {
        addTokenToLinks();
        addTokenToForms();
    });
    
    observer.observe(document.body, {
        childList: true,
        subtree: true
    });
    
})();

